<?php
require_once __DIR__ . '/url.php';

function seom_is_private_ip(string $ip): bool {
  if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) {
    $long = ip2long($ip);
    $ranges = [
      ['0.0.0.0', '0.255.255.255'],
      ['10.0.0.0', '10.255.255.255'],
      ['127.0.0.0', '127.255.255.255'],
      ['169.254.0.0', '169.254.255.255'],
      ['172.16.0.0', '172.31.255.255'],
      ['192.168.0.0', '192.168.255.255'],
      ['224.0.0.0', '239.255.255.255'],
    ];
    foreach ($ranges as [$start, $end]) {
      if ($long >= ip2long($start) && $long <= ip2long($end)) return true;
    }
    return false;
  }
  // IPv6: block link-local, loopback, unique local
  if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6)) {
    return (
      $ip === '::1' ||
      str_starts_with($ip, 'fe80:') ||
      str_starts_with($ip, 'fc') ||
      str_starts_with($ip, 'fd')
    );
  }
  return true;
}

function seom_validate_target_url(string $url, array $settings = []): array {
  $normalized = seom_normalize_url($url, ['allow_query_params' => (bool)($settings['allow_query_params'] ?? false)]);
  if (!$normalized) return ['ok' => false, 'error' => 'Invalid URL'];

  $parts = parse_url($normalized);
  $host = $parts['host'] ?? '';
  if ($host === '' || $host === 'localhost') return ['ok' => false, 'error' => 'Host not allowed'];

  // Resolve and block private IPs
  $resolved = gethostbyname($host);
  if ($resolved && filter_var($resolved, FILTER_VALIDATE_IP)) {
    if (seom_is_private_ip($resolved)) {
      return ['ok' => false, 'error' => 'Private/internal hosts are not allowed'];
    }
  }

  return ['ok' => true, 'url' => $normalized];
}
