<?php

function seom_normalize_url(string $url, array $opts = []): ?string {
  $url = trim($url);
  if ($url === '') return null;

  // Add scheme if missing
  if (!preg_match('~^https?://~i', $url)) {
    $url = 'https://' . ltrim($url, '/');
  }

  $parts = parse_url($url);
  if (!$parts || empty($parts['host'])) return null;

  $scheme = strtolower($parts['scheme'] ?? 'https');
  if ($scheme !== 'http' && $scheme !== 'https') return null;

  $host = strtolower($parts['host']);
  $port = isset($parts['port']) ? ':' . (int)$parts['port'] : '';
  $path = $parts['path'] ?? '/';
  if ($path === '') $path = '/';

  // Collapse multiple slashes
  $path = preg_replace('~//+~', '/', $path);

  $allowQuery = (bool)($opts['allow_query_params'] ?? false);
  $query = '';
  if ($allowQuery && !empty($parts['query'])) {
    // Sort query params for stable hashing
    parse_str($parts['query'], $q);
    ksort($q);
    $query = http_build_query($q);
    $query = $query === '' ? '' : '?' . $query;
  }

  // Remove fragment always
  $normalized = $scheme . '://' . $host . $port . $path . $query;

  // Optionally strip trailing slash (keep for root)
  $stripTrailing = (bool)($opts['strip_trailing_slash'] ?? false);
  if ($stripTrailing && $path !== '/' && str_ends_with($normalized, '/')) {
    $normalized = rtrim($normalized, '/');
  }

  return $normalized;
}

function seom_url_hash(string $url): string {
  return sha1($url);
}

function seom_is_same_host(string $url, string $baseHost, bool $includeSubdomains = false): bool {
  $h = parse_url($url, PHP_URL_HOST);
  if (!$h) return false;
  $h = strtolower($h);
  $baseHost = strtolower($baseHost);
  if ($h === $baseHost) return true;
  if ($includeSubdomains && str_ends_with($h, '.' . $baseHost)) return true;
  return false;
}

function seom_is_blocked_scheme(string $url): bool {
  return (bool)preg_match('~^(mailto:|tel:|javascript:)~i', trim($url));
}

function seom_resolve_url(string $href, string $baseUrl): ?string {
  $href = trim($href);
  if ($href === '' || $href === '#') return null;
  if (seom_is_blocked_scheme($href)) return null;

  // Absolute
  if (preg_match('~^https?://~i', $href)) return $href;

  // Protocol-relative
  if (str_starts_with($href, '//')) {
    $scheme = parse_url($baseUrl, PHP_URL_SCHEME) ?: 'https';
    return $scheme . ':' . $href;
  }

  $baseParts = parse_url($baseUrl);
  if (!$baseParts || empty($baseParts['host'])) return null;
  $scheme = $baseParts['scheme'] ?? 'https';
  $host = $baseParts['host'];
  $port = isset($baseParts['port']) ? ':' . (int)$baseParts['port'] : '';
  $basePath = $baseParts['path'] ?? '/';

  // Root-relative
  if (str_starts_with($href, '/')) {
    return $scheme . '://' . $host . $port . $href;
  }

  // Relative path
  $dir = preg_replace('~/[^/]*$~', '/', $basePath);
  $combined = $dir . $href;
  // Normalize ../ and ./
  $segments = [];
  foreach (explode('/', $combined) as $seg) {
    if ($seg === '' || $seg === '.') continue;
    if ($seg === '..') {
      array_pop($segments);
      continue;
    }
    $segments[] = $seg;
  }
  $path = '/' . implode('/', $segments);
  return $scheme . '://' . $host . $port . $path;
}

function seom_is_private_or_local_host(string $host): bool {
  $host = strtolower(trim($host));
  if ($host === 'localhost') return true;
  if ($host === '127.0.0.1' || $host === '::1') return true;

  // If it's an IP, check private ranges
  if (filter_var($host, FILTER_VALIDATE_IP)) {
    return seom_is_private_ip($host);
  }

  // Prevent obvious internal hostnames
  if (str_ends_with($host, '.local') || str_ends_with($host, '.internal')) return true;

  return false;
}

function seom_is_private_ip(string $ip): bool {
  if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) {
    $long = ip2long($ip);
    // 10.0.0.0/8
    if (($long & 0xff000000) === 0x0a000000) return true;
    // 172.16.0.0/12
    if (($long & 0xfff00000) === 0xac100000) return true;
    // 192.168.0.0/16
    if (($long & 0xffff0000) === 0xc0a80000) return true;
    // 127.0.0.0/8
    if (($long & 0xff000000) === 0x7f000000) return true;
    // 169.254.0.0/16
    if (($long & 0xffff0000) === 0xa9fe0000) return true;
    return false;
  }
  if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6)) {
    // fc00::/7 (ULA), fe80::/10 (link-local)
    return str_starts_with($ip, 'fc') || str_starts_with($ip, 'fd') || str_starts_with($ip, 'fe80') || $ip === '::1';
  }
  return true;
}
