<?php
require_once __DIR__ . '/../helpers/logger.php';
require_once __DIR__ . '/../helpers/url.php';
require_once __DIR__ . '/../helpers/security.php';

class AuditModel {
  private mysqli $db;
  public function __construct(mysqli $db) { $this->db = $db; }

  public function create(string $startUrl, array $settings, ?int $userId = null): array {
    $startUrlNorm = seom_normalize_url($startUrl, [
      'allow_query_params' => (bool)($settings['allow_query_params'] ?? false)
    ]);
    if (!$startUrlNorm) {
      return ['ok' => false, 'error' => 'Invalid URL.'];
    }

    $parts = parse_url($startUrlNorm);
    $host = strtolower($parts['host'] ?? '');
    if ($host === '' || $host === 'localhost') {
      return ['ok' => false, 'error' => 'Invalid host.'];
    }

    // SSRF protection: resolve and block private ranges
    $ip = gethostbyname($host);
    if (!$ip || seom_is_private_ip($ip)) {
      return ['ok' => false, 'error' => 'Host not allowed.'];
    }

    $domain = $host;
    $now = date('Y-m-d H:i:s');
    $settingsJson = json_encode($settings, JSON_UNESCAPED_SLASHES);

    $stmt = $this->db->prepare(
      "INSERT INTO audits (user_id, domain, start_url, status, settings, created_at, updated_at) VALUES (?, ?, ?, 'queued', ?, ?, ?)"
    );
    if (!$stmt) return ['ok' => false, 'error' => 'DB error'];
    $uid = $userId;
    $stmt->bind_param('isssss', $uid, $domain, $startUrlNorm, $settingsJson, $now, $now);
    if (!$stmt->execute()) {
      $stmt->close();
      return ['ok' => false, 'error' => 'DB insert failed'];
    }
    $auditId = (int)$stmt->insert_id;
    $stmt->close();

    return ['ok' => true, 'audit_id' => $auditId, 'domain' => $domain, 'start_url' => $startUrlNorm];
  }

  public function setStatus(int $auditId, string $status, ?string $errorMessage = null): bool {
    $now = date('Y-m-d H:i:s');
    $stmt = $this->db->prepare("UPDATE audits SET status=?, error_message=?, updated_at=? WHERE id=?");
    if (!$stmt) return false;
    $stmt->bind_param('sssi', $status, $errorMessage, $now, $auditId);
    $ok = $stmt->execute();
    $stmt->close();
    return (bool)$ok;
  }

  public function markStarted(int $auditId): bool {
    $now = date('Y-m-d H:i:s');
    $stmt = $this->db->prepare("UPDATE audits SET started_at=COALESCE(started_at, ?), updated_at=? WHERE id=?");
    if (!$stmt) return false;
    $stmt->bind_param('ssi', $now, $now, $auditId);
    $ok = $stmt->execute();
    $stmt->close();
    return (bool)$ok;
  }

  public function incrementDiscovered(int $auditId, int $count): void {
    $stmt = $this->db->prepare("UPDATE audits SET pages_discovered = pages_discovered + ?, updated_at=? WHERE id=?");
    if (!$stmt) return;
    $now = date('Y-m-d H:i:s');
    $stmt->bind_param('isi', $count, $now, $auditId);
    $stmt->execute();
    $stmt->close();
  }

  public function incrementCrawled(int $auditId, int $count): void {
    $stmt = $this->db->prepare("UPDATE audits SET pages_crawled = pages_crawled + ?, updated_at=? WHERE id=?");
    if (!$stmt) return;
    $now = date('Y-m-d H:i:s');
    $stmt->bind_param('isi', $count, $now, $auditId);
    $stmt->execute();
    $stmt->close();
  }

  public function getById(int $auditId): ?array {
    $stmt = $this->db->prepare("SELECT * FROM audits WHERE id=? LIMIT 1");
    if (!$stmt) return null;
    $stmt->bind_param('i', $auditId);
    $stmt->execute();
    $res = $stmt->get_result();
    $row = $res ? $res->fetch_assoc() : null;
    $stmt->close();
    return $row ?: null;
  }

  public function getNextRunnable(): ?array {
    // Prefer discovering/running audits first; then queued
    $sql = "SELECT * FROM audits WHERE status IN ('discovering','running','queued') ORDER BY FIELD(status,'discovering','running','queued'), created_at ASC LIMIT 1";
    $res = $this->db->query($sql);
    if (!$res) return null;
    $row = $res->fetch_assoc();
    return $row ?: null;
  }
}
