<?php
require_once __DIR__ . '/../helpers/url.php';

class UrlModel {
  private mysqli $db;
  public function __construct(mysqli $db) { $this->db = $db; }

  public function enqueue(int $auditId, string $url, int $depth = 0, string $source = 'internal', ?string $discoveredFrom = null, array $opts = []): bool {
    $norm = seom_normalize_url($url, [
      'allow_query_params' => (bool)($opts['allow_query_params'] ?? false),
      'strip_trailing_slash' => true,
    ]);
    if (!$norm) return false;
    $hash = sha1($norm);
    $now = date('Y-m-d H:i:s');

    $stmt = $this->db->prepare(
      "INSERT IGNORE INTO audit_urls (audit_id, url, url_hash, depth, discovered_from, source, state, created_at, updated_at) 
       VALUES (?, ?, ?, ?, ?, ?, 'queued', ?, ?)"
    );
    if (!$stmt) return false;
    $stmt->bind_param('ississss', $auditId, $norm, $hash, $depth, $discoveredFrom, $source, $now, $now);
    $ok = $stmt->execute();
    $stmt->close();
    return (bool)$ok;
  }

  public function nextBatch(int $auditId, int $batchSize): array {
    $batchSize = max(1, min(50, $batchSize));
    $stmt = $this->db->prepare(
      "SELECT id, url, depth FROM audit_urls WHERE audit_id=? AND state='queued' ORDER BY id ASC LIMIT ?"
    );
    $stmt->bind_param('ii', $auditId, $batchSize);
    $stmt->execute();
    $res = $stmt->get_result();
    $rows = [];
    while ($r = $res->fetch_assoc()) $rows[] = $r;
    $stmt->close();
    return $rows;
  }

  public function setState(int $urlId, string $state, array $fields = []): void {
    $allowed = ['queued','fetching','fetched','failed','skipped'];
    if (!in_array($state, $allowed, true)) return;

    $sets = ['state=?', 'updated_at=?'];
    $params = [$state, date('Y-m-d H:i:s')];
    $types = 'ss';

    $map = [
      'http_code' => 'i',
      'final_url' => 's',
      'redirect_hops' => 'i',
      'content_type' => 's',
      'load_time_ms' => 'i',
      'bytes' => 'i',
      'last_error' => 's',
      'fetched_at' => 's',
    ];

    foreach ($fields as $k => $v) {
      if (!array_key_exists($k, $map)) continue;
      $sets[] = "$k=?";
      $types .= $map[$k];
      $params[] = $v;
    }

    $sql = "UPDATE audit_urls SET " . implode(', ', $sets) . " WHERE id=?";
    $types .= 'i';
    $params[] = $urlId;

    $stmt = $this->db->prepare($sql);
    if (!$stmt) return;
    $stmt->bind_param($types, ...$params);
    $stmt->execute();
    $stmt->close();
  }

  public function counts(int $auditId): array {
    $sql = "SELECT 
      SUM(state='queued') AS queued,
      SUM(state='fetched') AS fetched,
      SUM(state='failed') AS failed,
      SUM(state='skipped') AS skipped,
      COUNT(*) AS total
      FROM audit_urls WHERE audit_id=?";
    $stmt = $this->db->prepare($sql);
    $stmt->bind_param('i', $auditId);
    $stmt->execute();
    $res = $stmt->get_result();
    $row = $res->fetch_assoc() ?: [];
    $stmt->close();
    return $row;
  }
}
