<?php
$config = require __DIR__ . '/../app/config/config.php';
?><!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width,initial-scale=1">
  <title><?php echo htmlspecialchars($config['app']['name']); ?> - Phase 1</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
</head>
<body class="bg-light">
<div class="container py-5">
  <div class="row justify-content-center">
    <div class="col-lg-8">
      <div class="card shadow-sm">
        <div class="card-body p-4">
          <h1 class="h3 mb-2">SEO Master – Cyber Bot (Phase 1)</h1>
          <p class="text-muted mb-4">Start an audit (crawl + basic parse) and watch progress. Scoring & issues will come in Phase 2.</p>

          <form id="auditForm" class="row g-3">
            <div class="col-12">
              <label class="form-label">Website URL</label>
              <input type="url" class="form-control" name="start_url" placeholder="https://example.com" required>
            </div>
            <div class="col-md-4">
              <label class="form-label">Max URLs</label>
              <input type="number" class="form-control" name="max_urls" value="300" min="10" max="5000">
            </div>
            <div class="col-md-4">
              <label class="form-label">Max Depth</label>
              <input type="number" class="form-control" name="max_depth" value="4" min="0" max="20">
            </div>
            <div class="col-md-4">
              <label class="form-label">Batch Size</label>
              <input type="number" class="form-control" name="batch_size" value="15" min="1" max="50">
            </div>
            <div class="col-12 d-flex gap-2">
              <button class="btn btn-primary" type="submit">Start Audit</button>
              <button class="btn btn-outline-secondary" type="button" id="btnStop" disabled>Stop Polling</button>
            </div>
          </form>

          <hr class="my-4">

          <div id="statusBox" class="d-none">
            <div class="d-flex align-items-center justify-content-between">
              <div>
                <div class="fw-semibold" id="statusTitle">Audit</div>
                <div class="text-muted small" id="statusMeta"></div>
              </div>
              <span class="badge bg-secondary" id="statusBadge">queued</span>
            </div>

            <div class="mt-3">
              <div class="d-flex justify-content-between small mb-1">
                <span>Progress</span>
                <span id="statusCounts">0/0</span>
              </div>
              <div class="progress">
                <div class="progress-bar" id="statusBar" style="width:0%"></div>
              </div>
            </div>

            <div class="mt-3 small text-muted" id="statusHint">
              Note: The crawler runs via cron batches. If you don’t see progress, confirm your cPanel cron is set.
            </div>
          </div>

          <div id="msg" class="mt-3"></div>

        </div>
      </div>

      <div class="text-center text-muted small mt-3">
        Phase 1 builds DB + crawler + parser. Dashboard scoring/issues are Phase 2.
      </div>
    </div>
  </div>
</div>

<script>
const form = document.getElementById('auditForm');
const msg = document.getElementById('msg');
const statusBox = document.getElementById('statusBox');
const statusBadge = document.getElementById('statusBadge');
const statusTitle = document.getElementById('statusTitle');
const statusMeta = document.getElementById('statusMeta');
const statusBar = document.getElementById('statusBar');
const statusCounts = document.getElementById('statusCounts');
const btnStop = document.getElementById('btnStop');

let pollTimer = null;
let currentAuditId = null;

function setMsg(html, type='info') {
  msg.innerHTML = `<div class="alert alert-${type} py-2 mb-0">${html}</div>`;
}

async function postForm(url, data) {
  const res = await fetch(url, { method:'POST', body: data });
  return await res.json();
}

async function getJson(url) {
  const res = await fetch(url);
  return await res.json();
}

function startPolling(auditId) {
  currentAuditId = auditId;
  statusBox.classList.remove('d-none');
  btnStop.disabled = false;

  if (pollTimer) clearInterval(pollTimer);
  pollTimer = setInterval(async () => {
    const s = await getJson(`../app/api/audits-status.php?audit_id=${auditId}`);
    if (!s.ok) return;

    statusTitle.textContent = `${s.domain}`;
    statusMeta.textContent = `Start: ${s.start_url}`;
    statusBadge.textContent = s.status;
    statusBadge.className = 'badge ' + (
      s.status === 'done' ? 'bg-success' :
      s.status === 'failed' ? 'bg-danger' :
      s.status === 'running' ? 'bg-primary' :
      'bg-secondary'
    );

    const discovered = Number(s.pages_discovered || 0);
    const crawled = Number(s.pages_crawled || 0);
    const pct = discovered > 0 ? Math.min(100, Math.round((crawled/discovered)*100)) : 0;
    statusBar.style.width = pct + '%';
    statusBar.textContent = pct + '%';
    statusCounts.textContent = `${crawled}/${discovered}`;

    if (s.status === 'done' || s.status === 'failed') {
      clearInterval(pollTimer);
      pollTimer = null;
      btnStop.disabled = true;
      setMsg(`Audit finished with status: <b>${s.status}</b>. (Phase 2 will show issues & score.)`, s.status==='done'?'success':'danger');
    }
  }, 2000);
}

btnStop.addEventListener('click', () => {
  if (pollTimer) clearInterval(pollTimer);
  pollTimer = null;
  btnStop.disabled = true;
  setMsg('Polling stopped. The crawler can still continue in cron.', 'warning');
});

form.addEventListener('submit', async (e) => {
  e.preventDefault();
  msg.innerHTML = '';
  const fd = new FormData(form);
  const res = await postForm('../app/api/audits-create.php', fd);
  if (!res.ok) {
    setMsg(res.message || 'Failed', 'danger');
    return;
  }
  setMsg(`Audit created. ID: <b>${res.audit_id}</b>. Waiting for cron crawl batches...`, 'success');
  startPolling(res.audit_id);
});
</script>
</body>
</html>
